# ideas/views.py

# ==============================================================================
# ۱. وارد کردن کتابخانه‌ها (بخش مرتب‌شده)
# ==============================================================================
import csv
from django.http import HttpResponse
from django_filters.rest_framework import DjangoFilterBackend

from rest_framework import viewsets, permissions, status, filters
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.views import APIView

from .models import (
    User, Tag, Campaign, Category, Idea, Comment, Attachment, Notification
)
from .permissions import IsAdmin, IsOwnerOrAdmin
from .serializers import (
    UserSerializer, TagSerializer, CampaignSerializer, CategorySerializer,
    IdeaListSerializer, IdeaDetailSerializer, CommentSerializer, AttachmentSerializer,
    NotificationSerializer
)

# ==============================================================================
# ۲. ViewSet ها
# ==============================================================================

class UserViewSet(viewsets.ModelViewSet):
    """ ViewSet برای مدیریت کاربران (فقط ادمین) """
    queryset = User.objects.all().order_by('id')
    serializer_class = UserSerializer
    permission_classes = [IsAdmin] # <<< اصلاح شد: از کلاس صحیح استفاده شد


class CategoryViewSet(viewsets.ModelViewSet):
    """ ViewSet برای مدیریت دسته‌بندی‌ها (فقط ادمین) """
    queryset = Category.objects.all().order_by('name')
    serializer_class = CategorySerializer
    permission_classes = [IsAdmin]


class CampaignViewSet(viewsets.ModelViewSet):
    """ ViewSet برای مدیریت کمپین‌ها """
    queryset = Campaign.objects.all().order_by('-start_date')
    serializer_class = CampaignSerializer

    def get_permissions(self):
        if self.action in ['create', 'update', 'partial_update', 'destroy']:
            self.permission_classes = [IsAdmin]
        else:
            self.permission_classes = [permissions.IsAuthenticated]
        return super().get_permissions()

    @action(detail=True, methods=['get'], permission_classes=[IsAdmin])
    def export_ideas(self, request, pk=None):
        campaign = self.get_object()
        response = HttpResponse(content_type='text/csv')
        response['Content-Disposition'] = f'attachment; filename="ideas_campaign_{campaign.pk}.csv"'

        writer = csv.writer(response)
        writer.writerow(['ID', 'Title', 'Author', 'Stage', 'Net Score', 'TRL', 'Submitted At'])
        
        for idea in campaign.ideas.all().select_related('author'):
            writer.writerow([
                idea.pk, idea.title, idea.author.username, idea.get_stage_display(),
                idea.score, idea.trl,
                idea.last_stage_change.strftime('%Y-%m-%d %H:%M') if idea.last_stage_change else ''
            ])
        return response


class IdeaViewSet(viewsets.ModelViewSet):
    """ ViewSet برای مدیریت ایده‌ها با قابلیت‌های کامل """
    queryset = Idea.objects.all().prefetch_related('likes', 'dislikes').select_related('author', 'campaign')    
    # <<< اضافه شد: بک‌اند فیلتر و جستجو باید مشخص شود
    filter_backends = [DjangoFilterBackend, filters.SearchFilter, filters.OrderingFilter]
    filterset_fields = {
    'stage': ['exact'],
    'campaign': ['exact'],
    'author': ['exact'],
    'campaign__category': ['exact'], # <<< این خط جادویی است
}
    search_fields = ['title', 'description']
    ordering_fields = ['created_at', 'score', 'last_stage_change']

    def get_serializer_class(self):
        if self.action == 'list':
            return IdeaListSerializer
        return IdeaDetailSerializer

    def get_permissions(self):
        # این متد قوانین دسترسی را بر اساس نوع درخواست (action) تنظیم می‌کند
        if self.action in ['update', 'partial_update', 'destroy']:
            self.permission_classes = [IsOwnerOrAdmin]
        else:
            # برای بقیه موارد (create, list, retrieve, like, dislike...) کاربر فقط باید لاگین کرده باشد
            # دسترسی ادمین برای change_stage در خود action تعریف شده
            self.permission_classes = [permissions.IsAuthenticated]
        return super().get_permissions()

    def perform_create(self, serializer):
        serializer.save(author=self.request.user)

    def get_serializer_context(self):
        return {'request': self.request}

    @action(detail=True, methods=['post'], permission_classes=[IsAdmin])
    def change_stage(self, request, pk=None):
        idea = self.get_object()
        new_stage = request.data.get('new_stage')
        if not new_stage or new_stage not in dict(Idea.STAGE_CHOICES):
            return Response({"detail": "Invalid or missing 'new_stage'"}, status=status.HTTP_400_BAD_REQUEST)
        
        idea.stage = new_stage
        idea.save()
        return Response(IdeaDetailSerializer(idea, context=self.get_serializer_context()).data)

    @action(detail=True, methods=['post'])
    def like(self, request, pk=None):
        idea = self.get_object()
        user = request.user
        
        if user in idea.likes.all():
            idea.likes.remove(user)
        else:
            idea.likes.add(user)
            idea.dislikes.remove(user) # اگر دیسلایک کرده بود، حذف می‌شود
        
        idea.save() # save() را می‌توانیم به بعد از محاسبه score منتقل کنیم
        return Response(IdeaDetailSerializer(idea, context=self.get_serializer_context()).data)

    @action(detail=True, methods=['post'])
    def dislike(self, request, pk=None):
        idea = self.get_object()
        user = request.user
        
        if user in idea.dislikes.all():
            idea.dislikes.remove(user)
        else:
            idea.dislikes.add(user)
            idea.likes.remove(user) # اگر لایک کرده بود، حذف می‌شود

        idea.save()
        return Response(IdeaDetailSerializer(idea, context=self.get_serializer_context()).data)


class CommentViewSet(viewsets.ModelViewSet):
    """ ViewSet برای کامنت‌های یک ایده خاص """
    serializer_class = CommentSerializer
    
    def get_queryset(self):
        return Comment.objects.filter(idea_id=self.kwargs['idea_pk']).order_by('-created_at')

    def get_permissions(self):
        if self.action in ['update', 'partial_update', 'destroy']:
            self.permission_classes = [IsOwnerOrAdmin]
        else:
            self.permission_classes = [permissions.IsAuthenticated]
        return super().get_permissions()

    def perform_create(self, serializer):
        # ۱. پیدا کردن ایده‌ای که روی آن کامنت گذاشته شده
        idea = Idea.objects.get(pk=self.kwargs['idea_pk'])
        # ۲. ذخیره کردن کامنت جدید با اطلاعات نویسنده و ایده
        comment=serializer.save(author=self.request.user, idea=idea)
        # ۳. ساختن نوتیفیکیشن (بخش جدید و مهم)
    # ما فقط در صورتی نوتیفیکیشن می‌سازیم که کامنت‌گذار، خود صاحب ایده نباشد
        if idea.author != self.request.user:
            Notification.objects.create(
                recipient=idea.author,         # گیرنده: صاحب ایده
                actor=self.request.user,       # عامل: کسی که کامنت گذاشته
                verb='commented on your idea', # فعل: روی ایده‌ی شما کامنت گذاشت
                target=idea                    # هدف: خود ایده
            )
        

class AttachmentViewSet(viewsets.ModelViewSet):
    """ ViewSet برای پیوست‌های یک ایده خاص """
    serializer_class = AttachmentSerializer
    
    def get_queryset(self):
        return Attachment.objects.filter(idea_id=self.kwargs['idea_pk'])
        
    def get_permissions(self):
        if self.action == 'destroy':
            self.permission_classes = [IsOwnerOrAdmin] 
        else:
            self.permission_classes = [permissions.IsAuthenticated]
        return super().get_permissions()

    def perform_create(self, serializer):
        idea = Idea.objects.get(pk=self.kwargs['idea_pk'])
        file_name = self.request.data.get('file').name
        file_type = file_name.split('.')[-1].lower() if '.' in file_name else 'unknown'
        serializer.save(idea=idea, file_type=file_type)


class TagViewSet(viewsets.ModelViewSet):
    """ ViewSet برای مدیریت تگ‌ها """
    queryset = Tag.objects.all().order_by('name')
    serializer_class = TagSerializer
    
    def get_permissions(self):
        if self.action in ['list', 'retrieve']:
            self.permission_classes = [permissions.IsAuthenticated]
        else:
            self.permission_classes = [IsAdmin]
        return super().get_permissions()

# ==============================================================================
# ۳. API View های سفارشی
# ==============================================================================

class AnalyticsSummaryAPIView(APIView):
    """ یک API View برای نمایش آمارهای کلی سیستم """
    permission_classes = [permissions.IsAuthenticated]

    def get(self, request, format=None):
        total_ideas = Idea.objects.count()
        active_campaigns = Campaign.objects.filter(status='active').count()
        implemented_ideas = Idea.objects.filter(stage='implemented').count()
        satisfaction_rate = (implemented_ideas / total_ideas * 100) if total_ideas > 0 else 0

        funnel_stats = {
            stage_display: Idea.objects.filter(stage=stage_key).count()
            for stage_key, stage_display in Idea.STAGE_CHOICES
        }

        return Response({
            'total_ideas': total_ideas,
            'active_campaigns': active_campaigns,
            'satisfaction_rate': round(satisfaction_rate, 2),
            'funnel_pipeline': funnel_stats
        })
        
# ==============================================================================
# ۴. ViewSet نوتیفیکیشن (جدید)
# ==============================================================================
class NotificationViewSet(viewsets.ReadOnlyModelViewSet):
    """
    یک ViewSet فقط-خواندنی برای نمایش نوتیفیکیشن‌های کاربر.
    """
    serializer_class = NotificationSerializer
    permission_classes = [permissions.IsAuthenticated]

    def get_queryset(self):
        """
        این متد تضمین می‌کند که هر کاربر فقط نوتیفیکیشن‌های خودش را می‌بیند.
        """
        # ما از 'related_name' که در مدل تعریف کردیم ('notifications') استفاده می‌کنیم.
        return self.request.user.notifications.all()

    @action(detail=False, methods=['post'])
    def mark_all_as_read(self, request):
        """
        یک اکشن سفارشی برای 'خوانده شده' کردن تمام نوتیفیکیشن‌ها.
        آدرس: POST /api/notifications/mark_all_as_read/
        """
        self.request.user.notifications.update(unread=False)
        return Response(status=status.HTTP_204_NO_CONTENT)