# ideas/tests.py

from rest_framework.test import APITestCase
from rest_framework import status
from django.urls import reverse
from .models import User

class UserRegistrationAPITest(APITestCase):
    """
    تست برای API ثبت‌نام کاربر.
    """

    # این متد، تست اصلی ماست
    def test_register_user_successfully(self):
        """
        تضمین می‌کند که یک کاربر جدید می‌تواند با اطلاعات معتبر ثبت‌نام کند.
        """
        # ۱. آدرس URL برای ثبت‌نام رو پیدا می‌کنیم
        # 'user-list' نامی است که روتر به صورت خودکار برای UserViewSet می‌سازد
        url = reverse('user-list') 

        # ۲. اطلاعاتی که می‌خواهیم برای ثبت‌نام بفرستیم
        data = {
            'username': 'testuser',
            'email': 'test@example.com',
            'password': 'SomeStrongPassword123',
            're_password': 'SomeStrongPassword123'
        }

        # ۳. یک درخواست POST به آدرس ثبت‌نام می‌فرستیم
        response = self.client.post(url, data, format='json')

        # ۴. بررسی می‌کنیم که آیا همه چیز درست پیش رفته یا نه

        # آیا کد وضعیت '201 Created' برگردانده شده؟
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

        # آیا واقعاً یک کاربر جدید در دیتابیس ساخته شده؟
        self.assertEqual(User.objects.count(), 1)

        # آیا نام کاربری کاربر ساخته شده همان چیزی است که فرستادیم؟
        self.assertEqual(User.objects.get().username, 'testuser')
        
# --- کلاس تست جدید برای ایده‌ها ---
from .models import Idea, Campaign, Category

class IdeaAPITest(APITestCase):
    """
    تست برای API مربوط به ایده‌ها.
    """
    def setUp(self):
        self.user = User.objects.create_user(username='testuser', password='password123')
        self.category = Category.objects.create(name='Test Category')
        self.campaign = Campaign.objects.create(
            title='Test Campaign', description='A test campaign',
            category=self.category, start_date='2025-01-01', end_date='2025-12-31'
        )
        login_url = reverse('jwt-create')
        response = self.client.post(login_url, {'username': 'testuser', 'password': 'password123'}, format='json')
        self.client.credentials(HTTP_AUTHORIZATION=f'Bearer {response.data["access"]}')

    def test_create_idea_authenticated_user(self):
        url = reverse('idea-list')
        data = {'title': 'My awesome test idea', 'description': '...', 'campaign': self.campaign.id}
        response = self.client.post(url, data, format='json')
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Idea.objects.count(), 1)
        self.assertEqual(Idea.objects.get().author, self.user)

    def test_create_idea_unauthenticated_user(self):
        self.client.force_authenticate(user=None)
        url = reverse('idea-list')
        data = {'title': 'A forbidden idea', 'description': '...', 'campaign': self.campaign.id}
        response = self.client.post(url, data, format='json')
        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED)

    def test_update_idea_by_owner(self):
        idea = Idea.objects.create(author=self.user, campaign=self.campaign, title='Original Title', description='...')
        url = reverse('idea-detail', kwargs={'pk': idea.pk})
        updated_data = {'title': 'Updated Title'}
        response = self.client.patch(url, updated_data, format='json') # <<< اصلاح شد
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        idea.refresh_from_db()
        self.assertEqual(idea.title, 'Updated Title')

    def test_update_idea_by_other_user_fails(self):
        other_user = User.objects.create_user(username='otheruser', password='password123')
        idea = Idea.objects.create(author=other_user, campaign=self.campaign, title='Someone Else\'s Idea', description='...')
        url = reverse('idea-detail', kwargs={'pk': idea.pk})
        updated_data = {'title': 'Hacked Title'}
        response = self.client.patch(url, updated_data, format='json') # <<< اصلاح شد
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_update_idea_by_admin_succeeds(self):
        other_user = User.objects.create_user(username='otheruser', password='password123')
        idea = Idea.objects.create(author=other_user, campaign=self.campaign, title='Someone Else\'s Idea', description='...')
        admin_user = User.objects.create_superuser(username='admin', password='password123', email='admin@test.com')
        self.client.force_authenticate(user=admin_user)
        url = reverse('idea-detail', kwargs={'pk': idea.pk})
        updated_data = {'title': 'Admin Updated Title'}
        response = self.client.patch(url, updated_data, format='json') # <<< اصلاح شد
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        idea.refresh_from_db()
        self.assertEqual(idea.title, 'Admin Updated Title')