# ideas/serializers.py

from rest_framework import serializers
from djoser.serializers import UserCreateSerializer as DjoserUserCreateSerializer
from .models import User, Category, Tag, Campaign, Idea, Comment, Attachment
from .models import User, Category, Tag, Campaign, Idea, Comment, Attachment, Notification # <<< مطمئن شو Notification اینجاست
# ==============================================================================
# ۱. سریالایزرهای کاربر
# ==============================================================================

class UserCreateSerializer(DjoserUserCreateSerializer):
    class Meta(DjoserUserCreateSerializer.Meta):
        model = User
        fields = ('id', 'email', 'username', 'password', 'role')
        read_only_fields = ('role',)

    def create(self, validated_data):
        validated_data['role'] = 'user'
        return super().create(validated_data)

class UserSerializer(serializers.ModelSerializer):
    class Meta:
        model = User
        fields = ('id', 'username', 'email', 'role', 'is_active', 'first_name', 'last_name')
        read_only_fields = ('role',)

# ==============================================================================
# ۲. سریالایزرهای مدل‌های کمکی
# ==============================================================================

class CategorySerializer(serializers.ModelSerializer):
    class Meta:
        model = Category
        fields = ('id', 'name')

class TagSerializer(serializers.ModelSerializer):
    class Meta:
        model = Tag
        fields = ('id', 'name')

class AttachmentSerializer(serializers.ModelSerializer):
    class Meta:
        model = Attachment
        fields = ('id', 'file', 'file_type', 'uploaded_at')
        read_only_fields = ('uploaded_at',)

class CommentSerializer(serializers.ModelSerializer):
    author_username = serializers.CharField(source='author.username', read_only=True)

    class Meta:
        model = Comment
        fields = ('id', 'text', 'author', 'author_username', 'idea', 'created_at')
        read_only_fields = ('author', 'idea', 'created_at')

# ==============================================================================
# ۳. سریالایزرهای مدل‌های اصلی
# ==============================================================================

class CampaignSerializer(serializers.ModelSerializer):
    category_name = serializers.CharField(source='category.name', read_only=True)
    idea_count = serializers.SerializerMethodField()
    
    class Meta:
        model = Campaign
        fields = ('id', 'title', 'description', 'category', 'category_name', 
                  'start_date', 'end_date', 'status', 'visibility', 'idea_count')
        
    def get_idea_count(self, obj):
        return obj.ideas.count()


class IdeaListSerializer(serializers.ModelSerializer):
    """ سریالایزر خلاصه‌شده برای نمایش در لیست‌ها (فقط خواندنی) """
    author_username = serializers.CharField(source='author.username', read_only=True)
    campaign_title = serializers.CharField(source='campaign.title', read_only=True)
    comment_count = serializers.SerializerMethodField(read_only=True)
    net_score = serializers.SerializerMethodField(read_only=True)

    class Meta:
        model = Idea
        fields = ('id', 'title', 'author_username','campaign', 'campaign_title', 'stage', 
                  'net_score', 'comment_count', 'last_stage_change')

    def get_comment_count(self, obj):
        return obj.comments.count()
        
    def get_net_score(self, obj):
        # این فیلد دیگر در مدل وجود دارد، می‌توانیم مستقیما از آن استفاده کنیم
        return obj.score


class IdeaDetailSerializer(serializers.ModelSerializer):
    """ سریالایزر کامل برای ساختن، ویرایش و نمایش جزئیات یک ایده """
    author_username = serializers.CharField(source='author.username', read_only=True)
    comments = CommentSerializer(many=True, read_only=True)
    attachments = AttachmentSerializer(many=True, read_only=True)
    is_liked = serializers.SerializerMethodField()
    is_disliked = serializers.SerializerMethodField()

    # --- بخش حیاتی اصلاح شده ---
    # این فیلدها اجازه می‌دهند که هنگام ساخت/ویرایش، ID ارسال شود
    campaign = serializers.PrimaryKeyRelatedField(queryset=Campaign.objects.all())
    tags = serializers.PrimaryKeyRelatedField(many=True, queryset=Tag.objects.all(), required=False)

    class Meta:
        model = Idea
        fields = [
            'id', 'title', 'description', 'author_username', 'is_liked', 'is_disliked',
            'campaign',  # <<< این فیلد اکنون ورودی ID را قبول می‌کند
            'tags',
            'stage', 'score', 'last_stage_change',
            'utility', 'inventive_step', 'novelty_statement', 'trl',
            'comments', 'attachments'
        ]
        # فیلدهایی که کاربر نباید مستقیماً تغییر دهد
        read_only_fields = ('author_username', 'stage', 'score', 'last_stage_change')

    def get_is_liked(self, obj):
        user = self.context.get('request').user
        return user.is_authenticated and obj.likes.filter(id=user.id).exists()

    def get_is_disliked(self, obj):
        user = self.context.get('request').user
        return user.is_authenticated and obj.dislikes.filter(id=user.id).exists()

# ==============================================================================
# ۴. سریالایزر نوتیفیکیشن (جدید)
# ==============================================================================
class NotificationSerializer(serializers.ModelSerializer):
    # برای نمایش نام کاربری به جای ID
    actor_username = serializers.CharField(source='actor.username', read_only=True)

    # برای نمایش متن هدف (مثلاً عنوان ایده)
    target = serializers.StringRelatedField(read_only=True)

    class Meta:
        model = Notification
        fields = ('id', 'actor_username', 'verb', 'target', 'unread', 'timestamp')