from typing import Dict
from ideas.models import EvaluationCriterion, Idea
from rest_framework.exceptions import ValidationError



def calculate_final_score(scores: Dict[str, float]) -> float:
    """
    Calculate weighted final score based on active evaluation criteria.

    scores example (0–100):ظ
    {
        "strategic_alignment": 80,
        "economic_feasibility": 60,
        "novelty": 90
    }
    """

    criteria = EvaluationCriterion.objects.filter(is_active=True)

    if not criteria.exists():
        raise ValidationError("No active evaluation criteria defined.")

    weighted_sum = 0.0    # ✅ حتماً باید اینجا مقداردهی شود
    total_weight = 0.0

    for criterion in criteria:
        score = scores.get(criterion.key)

        # نبودن score = نادیده گرفته می‌شود (Design decision ✅)
        if score is None:
            continue

        if not isinstance(score, (int, float)):
            raise ValidationError(
                f"Score for '{criterion.key}' must be a number."
            )

        if not 0 <= score <= 100:
            raise ValidationError(
                f"Score for '{criterion.key}' must be between 0 and 100."
            )

        if criterion.weight <= 0:
            continue  # وزن صفر یا منفی بی‌اثر است

        weighted_sum += score * criterion.weight
        total_weight += criterion.weight

    if total_weight == 0:
        raise ValidationError(
            "No valid scores provided or all criteria weights are zero."
        )

    final_score = weighted_sum / total_weight
    return round(final_score, 2)

def apply_h_level_rules(idea: Idea) -> str | None:
    if idea.final_score is None:
        return None

    rules = decision.get(idea.h_level)
    if not rules:
        return None

    score = idea.final_score

    if score >= rules["next"]:
        idea.stage = "next"
        decision = "next"

    elif score >= rules["revise"]:
        idea.stage = "revise"
        decision = "revise"

    elif score >= rules["backlog"]:
        idea.stage = "backlog"
        decision = "backlog"

    else:
        idea.is_archived = True
        decision = "archive"

    idea.save()
    return decision