from rest_framework import serializers
from django.utils.timesince import timesince

from ideas.services.decision import calculate_decision
from ideas.services.scoring import calculate_final_score
from .models import ActivityLog, EvaluationCriterion, EvaluationResult, User, Category, Tag, Campaign, Idea, Comment, Attachment, Notification,RefinementQuestion,RefinementReview,RefinementAnswer
from django.db import transaction
from django.db.models import Avg

from ideas import models  
# --- 1. Basic Serializers (باید اول باشند چون بقیه از اینها استفاده می‌کنند) ---
class UserMiniSerializer(serializers.ModelSerializer):
    class Meta:
        model = User
        fields = ('id', 'username')

class RefinementQuestionSerializer(serializers.ModelSerializer):
    class Meta:
        model = RefinementQuestion
        fields = ('key', 'title')


class RefinementAnswerInputSerializer(serializers.Serializer):
    question_key = serializers.CharField()
    answer = serializers.BooleanField()
    comment = serializers.CharField(required=False, allow_blank=True)
class RefinementAnswerDisplaySerializer(serializers.ModelSerializer):
    question_key = serializers.CharField(source='question.key')
    question_title = serializers.CharField(source='question.title')

    class Meta:
        model = RefinementAnswer
        fields = ['question_key', 'question_title', 'answer', 'comment']
class RefinementReviewDisplaySerializer(serializers.ModelSerializer):
    answers = RefinementAnswerDisplaySerializer(many=True, read_only=True)
    reviewer_name = serializers.CharField(source='reviewer.get_full_name', read_only=True)

    class Meta:
        model = RefinementReview
        fields = ['id', 'is_passed', 'created_at', 'reviewer_name', 'answers']
class RefinementReviewSerializer(serializers.ModelSerializer):
    answers = RefinementAnswerDisplaySerializer(many=True, read_only=True)

    class Meta:
        model = RefinementReview
        fields = ['id', 'idea', 'reviewer', 'created_at', 'is_passed', 'answers']
    def get_fields(self):
        fields = super().get_fields()
        request = self.context.get("request")

        if request and request.user.is_authenticated:
            if not (
                request.user.is_staff or
                request.user.groups.filter(name="reviewer").exists()
            ):
                # یوزر معمولی
                for field in ["is_passed", "comment"]:
                    fields[field].read_only = True

        return fields


class RefinementReviewCreateSerializer(serializers.Serializer):
    answers = RefinementAnswerInputSerializer(many=True)

    def create(self, validated_data):
        idea = self.context["idea"]
        user = self.context["request"].user
        answers_data = validated_data["answers"]
        
        is_passed = all(a["answer"] is True for a in answers_data)
        
        with transaction.atomic():
            review, created = RefinementReview.objects.update_or_create(
                idea=idea,
                defaults={
                    'reviewer': user,
                    'is_passed': is_passed
                }
            )
            
            if not created:
                review.answers.all().delete()
            
            for a in answers_data:
                question, _ = RefinementQuestion.objects.get_or_create(
                    key=a["question_key"],
                    defaults={
                        "title": a["question_key"].replace("_", " ").title(),
                        "is_active": True,
                        "order": 0,
                    }
                )
                
                RefinementAnswer.objects.create(
                    review=review,
                    question=question,
                    answer=a["answer"],
                    comment=a.get("comment", ""),
                )
            
        return review  # ✅ این خط حیاتی است!

class CampaignMiniSerializer(serializers.ModelSerializer):
    class Meta:
        model = Campaign
        fields = ('id', 'title')

class UserSerializer(serializers.ModelSerializer):
    class Meta:
        model = User
        fields = ('id', 'username', 'email', 'role', 'is_active', 'first_name', 'last_name')
        read_only_fields = ('role',)

class CategorySerializer(serializers.ModelSerializer):
    class Meta:
        model = Category
        fields = ('id', 'name')

class TagSerializer(serializers.ModelSerializer):
    class Meta:
        model = Tag
        fields = ('id', 'name')


class AttachmentSerializer(serializers.ModelSerializer):
    filename = serializers.SerializerMethodField()
    url = serializers.SerializerMethodField()

    class Meta:
        model = Attachment
        fields = ['id', 'filename', 'url']

    def get_filename(self, obj):
        return obj.file.name.split('/')[-1]

    def get_url(self, obj):
        request = self.context.get("request")
        return request.build_absolute_uri(obj.file.url)

class CommentSerializer(serializers.ModelSerializer):
    author_username = serializers.CharField(source='author.username', read_only=True)

    class Meta:
        model = Comment
        fields = ['id', 'text', 'author_username', 'created_at', 'edited', 'edited_at']
        read_only_fields = ['author', 'created_at', 'edited', 'edited_at']
    
    def get_is_reviewer_comment(self, obj):
        return obj.author.role == 'reviewer'

# --- 2. Activity Log (این باید قبل از Idea باشد) ---
class ActivityLogSerializer(serializers.ModelSerializer):
    changedBy = serializers.CharField(source="changed_by.username", read_only=True)
    previousStage = serializers.CharField(source="previous_stage", read_only=True)
    newStage = serializers.CharField(source="new_stage", read_only=True)
    notes = serializers.CharField()
    changedAt = serializers.DateTimeField(source="changed_at", read_only=True)

    class Meta:
        model = ActivityLog
        fields = [
            "id",
            "previousStage",
            "newStage",
            "notes",
            "changedBy",
            "changedAt",
        ]

# --- 3. Campaign ---
class CampaignSerializer(serializers.ModelSerializer):
    category_name = serializers.CharField(source='category.name', read_only=True)
    idea_count = serializers.SerializerMethodField()

    class Meta:
        model = Campaign
        fields = [
            'id', 'title', 'description',
            'category', 'category_name',
            'start_date', 'end_date',
            'status', 'visibility',
            'idea_count'
        ]

    def get_idea_count(self, obj):
        return obj.ideas.filter(is_deleted=False).count()

# --- 4. Ideas (List) ---
class IdeaListSerializer(serializers.ModelSerializer):
    author_username = serializers.CharField(source='author.username', read_only=True)
    campaign_title = serializers.CharField(source='campaign.title', read_only=True)
    comment_count = serializers.IntegerField(source='comments.count', read_only=True)
    net_score = serializers.IntegerField(source='score', read_only=True)
    likes_count = serializers.SerializerMethodField(read_only=True)
    dislikes_count = serializers.SerializerMethodField(read_only=True)
    idea_type = serializers.CharField(read_only=False)
    class Meta:
        model = Idea
        fields = [
            'id',
            'title',
            'description',
            'author',
            'author_username',
            'campaign',
            'campaign_title',
            'stage',
            'score',
            'net_score',
            'likes_count',
            'dislikes_count',
            'comment_count',
            'last_stage_change',
            'tags',
            'idea_type',

            # ✅ FIX
            'is_archived',
        ]

    def get_likes_count(self, obj):
        return obj.likes.count()

    def get_dislikes_count(self, obj):
        return obj.dislikes.count()
    
    def get_comment_count(self, obj):
        return obj.comments.count()
# --- 4.5 Archived Ideas (Archive Page) ---
class ArchiveIdeaSerializer(serializers.ModelSerializer):
    campaign = CampaignMiniSerializer(read_only=True, allow_null=True)
    archived_by = UserMiniSerializer(read_only=True)

    class Meta:
        model = Idea
        fields = [
            'id',
            'title',
            'stage',
            'campaign',
            'archived_at',
            'archived_by',
        ]

# --- 5. Idea Detail (اصلی - با رفع تمام مشکلات) ---
class IdeaDetailSerializer(serializers.ModelSerializer):
    author_username = serializers.CharField(source='author.username', read_only=True)
    campaign_title = serializers.CharField(source='campaign.title', read_only=True)
    has_evaluated = serializers.SerializerMethodField()
    my_evaluation = serializers.SerializerMethodField()
    can_edit_evaluation = serializers.SerializerMethodField()

    # ✅ 1. Short Description (قابل ویرایش)
    short_description = serializers.CharField(source='description', required=False, allow_blank=True)

    # ✅ 2. Idea Type (قابل ویرایش - صراحتاً تعریف شد)
    # این باعث می‌شود جنگو مقادیر را چک کند و اجازه ذخیره بدهد
    idea_type = serializers.ChoiceField(choices=Idea.IDEA_TYPE_CHOICES, required=False)

    
    
    # --- Concept Development Fields ---

    detailed_description = serializers.CharField(
        required=False,
        allow_blank=True
    )

    strategic_alignment = serializers.CharField(
        required=False,
        allow_blank=True
    )

    risks = serializers.CharField(
        required=False,
        allow_blank=True
    )

    initial_people = serializers.CharField(
        required=False,
        allow_blank=True
    )

    initial_abilities = serializers.CharField(
        required=False,
        allow_blank=True
    )

    initial_departments = serializers.CharField(
        required=False,
        allow_blank=True
    )

    feasibility = serializers.CharField(
        required=False,
        allow_blank=True
    )

    trl = serializers.IntegerField(
        required=False,
        allow_null=True
    )
    
    
    
    
    
    
    # ✅ 3. Tags (قابل ویرایش - دریافت ID)
    tags = serializers.PrimaryKeyRelatedField(
        many=True, 
        queryset=Tag.objects.all(),
        required=False
    )

    # فیلدهای فقط خواندنی (چون ActivityLogSerializer الان بالاتر تعریف شده، اینجا ارور نمی‌دهد)
    attachment = serializers.FileField(
        write_only=True, required=False, allow_null=True
    )
    attachments = AttachmentSerializer(many=True, read_only=True)  # 👈 لازم برای آپلود

    comments = CommentSerializer(many=True, read_only=True)
    activity_logs = ActivityLogSerializer(many=True, read_only=True)

    # متدهای محاسباتی
    is_liked = serializers.SerializerMethodField()
    is_disliked = serializers.SerializerMethodField()
    likes_count = serializers.SerializerMethodField()
    dislikes_count = serializers.SerializerMethodField()
    comment_count = serializers.SerializerMethodField()
    liked_by = serializers.SerializerMethodField()
    disliked_by = serializers.SerializerMethodField()
    is_archived = serializers.BooleanField(read_only=True)

    class Meta:
        model = Idea
        fields = [
            'id', 'title', 
            'short_description', 
            'idea_type', # ✅ حتماً اینجا باشد
            'author_username',
            'campaign', 'campaign_title',
            'stage', 'score', 'created_at', 'last_stage_change',
            'tags','is_archived','archived_at',
            'archived_by',
            'attachments','attachment',
            'comments', 'comment_count',
            'activity_logs',
            'is_liked', 'is_disliked',
            'likes_count', 'dislikes_count',
            'liked_by', 'disliked_by',
            # concept
            'detailed_description',
            'strategic_alignment',
            'risks',
            'initial_people',
            'initial_abilities',
            'initial_departments',
            'feasibility',
            'trl',
            # other
            'edited',
            'edited_at',
            'edited_by',
            "has_evaluated",
            "can_edit_evaluation",
            "my_evaluation",
        ]
        read_only_fields = ('stage', 'score', 'author', 'created_at')
    def update(self, instance, validated_data):
        request = self.context.get('request')
        file_obj = None
        if request and hasattr(request, 'FILES'):
            file_obj = request.FILES.get('attachment')
        instance = super().update(instance, validated_data)

        if file_obj:
            Attachment.objects.create(
                idea=instance,
                file=file_obj,
                file_type=file_obj.name.split('.')[-1].lower()
            )
        return instance


    # ✅ تبدیل تگ‌ها به آبجکت کامل موقع نمایش
    def to_representation(self, instance):
        representation = super().to_representation(instance)
        representation['tags'] = TagSerializer(instance.tags.all(), many=True).data
        return representation

    # ---------- Methods ----------
    def get_is_liked(self, obj):
        user = self.context['request'].user
        return user.is_authenticated and obj.likes.filter(id=user.id).exists()

    def get_is_disliked(self, obj):
        user = self.context['request'].user
        return user.is_authenticated and obj.dislikes.filter(id=user.id).exists()

    def get_likes_count(self, obj):
        return obj.likes.count()

    def get_dislikes_count(self, obj):
        return obj.dislikes.count()

    def get_comment_count(self, obj):
        return obj.comments.count()

    def get_liked_by(self, obj):
        return list(obj.likes.values_list('username', flat=True))

    def get_disliked_by(self, obj):
        return list(obj.dislikes.values_list('username', flat=True))
    def get_has_evaluated(self, obj):
        request = self.context.get("request")
        user = request.user if request else None

        if not user or not user.is_authenticated:
            return False

        return obj.evaluations.filter(evaluator=user).exists()

    def get_my_evaluation(self, obj):
        request = self.context.get("request")
        user = request.user if request else None

        if not user or not user.is_authenticated:
            return None

        evaluation = obj.evaluations.filter(
            evaluator=user
        ).first()

        if not evaluation:
            return None
        decision = calculate_decision(
            evaluation.final_score,
            obj.h_level
        )
        return {
            "final_score": evaluation.final_score,
            "h_level": obj.h_level,
            "scores": evaluation.scores,
            "decision": decision,
        }

    def get_can_edit_evaluation(self, obj):
        request = self.context.get("request")
        user = request.user if request else None

        if not user or not user.is_authenticated:
            return False

        return user.is_staff or user.groups.filter(name="Reviewer").exists()



class NotificationActorSerializer(serializers.ModelSerializer):
    full_name = serializers.SerializerMethodField()

    class Meta:
        model = User
        fields = ("id", "full_name")

    def get_full_name(self, obj):
        name = obj.get_full_name()
        return name if name else obj.username


# --- 6. Notifications ---
class NotificationSerializer(serializers.ModelSerializer):
    actor = NotificationActorSerializer(read_only=True)
    type = serializers.SerializerMethodField()   # ✅ این
    target_type = serializers.SerializerMethodField()
    target_id = serializers.SerializerMethodField()
    target_title = serializers.SerializerMethodField()
    created_at_human = serializers.SerializerMethodField()
    unread = serializers.BooleanField()
    class Meta:
        model = Notification
        fields = (
            "id",
            "type",
            "actor",

            "target_type",
            "target_id",
            "target_title",

            "created_at_human",

            "unread",
        )

    def get_target_type(self, obj):
        if obj.target_content_type:
            return obj.target_content_type.model
        return None

    def get_target_id(self, obj):
        return obj.target_object_id

    def get_target_title(self, obj):
        if not obj.target_content_type or not obj.target_object_id:
            return None

        if obj.target_content_type.model == "idea":
            return (
                Idea.objects
                .filter(id=obj.target_object_id)
                .values_list("title", flat=True)
                .first()
            )

        if obj.target_content_type.model == "comment":
            comment = Comment.objects.filter(
                id=obj.target_object_id
            ).select_related("idea").first()

            if comment and comment.idea:
                return comment.idea.title

        return None

    def get_actor(self, obj):
        if not obj.actor:
            return None
        return {
            "id": obj.actor.id,
            "full_name": obj.actor.get_full_name() or obj.actor.username,
        }
    def get_type(self, obj):
        """
        Backend: verb
        Frontend: type
        """
        return obj.verb
    def get_created_at_human(self, obj):
        return timesince(obj.timestamp) + " ago"

# --- Mini Serializers (برای Archive Page) ---
class EvaluationCriterionSerializer(serializers.ModelSerializer):
    class Meta:
        model = EvaluationCriterion
        fields = ["key", "title", "weight"]
# در serializers.py
class EvaluateIdeaSerializer(serializers.Serializer):
    scores = serializers.DictField(
        child=serializers.IntegerField(min_value=1, max_value=100)
    )

    def create(self, validated_data):
        idea = validated_data["idea"]

        final_score = calculate_final_score(...)

        idea.final_score = final_score
        idea.save(update_fields=["final_score"])


        EvaluationResult.objects.create(
            idea=idea,
            evaluator=self.context["request"].user,
            final_score=final_score,
        )
        decision = calculate_decision(final_score)

        agg = idea.evaluations.aggregate(avg=Avg("final_score"))
        idea.final_score = agg["avg"]
        return {
            "final_score": final_score,
            "decision": decision,
        }
        # ---------- update Idea ----------

class UpdateEvaluationSerializer(serializers.ModelSerializer):
    class Meta:
        model = EvaluationResult
        fields = ["scores"]

    def update(self, instance, validated_data):
        instance.scores = validated_data["scores"]
        instance.save()
        instance.final_score = calculate_final_score(instance.scores)

        # محاسبه نمره نهایی دوباره
        final_score = calculate_final_score(instance.scores)
        instance.final_score = final_score
        instance.save()

        # اعمال قوانین H-Level
        apply_decision(instance.idea)

        return instance
class MyEvaluationSerializer(serializers.Serializer):
    final_score = serializers.FloatField()
    h_level = serializers.CharField()
    scores = serializers.DictField(child=serializers.IntegerField())

class NotificationActorSerializer(serializers.ModelSerializer):
    full_name = serializers.CharField()

    class Meta:
        model = User
        fields = ("id", "full_name")
