from django.db import models
from django.contrib.auth.models import AbstractUser
from django.conf import settings
from django.utils import timezone
from django.contrib.contenttypes.models import ContentType
from django.contrib.contenttypes.fields import GenericForeignKey


# 1) User
class User(AbstractUser):
    ROLE_CHOICES = (
        ('admin', 'Admin'),
        ('reviewer', 'Reviewer'),
        ('user', 'User'),
    )
    role = models.CharField(max_length=20, choices=ROLE_CHOICES, default='user')


# 2) Helpers
class Category(models.Model):
    name = models.CharField(max_length=100, unique=True)
    def __str__(self):
        return self.name


class Tag(models.Model):
    name = models.CharField(max_length=50, unique=True)
    def __str__(self):
        return self.name


# 3) Soft-delete mixin
class SoftDeleteMixin(models.Model):
    is_deleted = models.BooleanField(default=False)
    deleted_at = models.DateTimeField(null=True, blank=True)
    deleted_by = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        null=True, blank=True,
        on_delete=models.SET_NULL,
        related_name='deleted_%(class)ss'
    )

    class Meta:
        abstract = True

    def soft_delete(self, user=None):
        self.is_deleted = True
        self.deleted_at = timezone.now()
        if user and getattr(user, 'is_authenticated', False):
            self.deleted_by = user
        self.save(update_fields=['is_deleted', 'deleted_at', 'deleted_by'])

    def restore(self):
        self.is_deleted = False
        self.deleted_at = None
        self.deleted_by = None
        self.save(update_fields=['is_deleted', 'deleted_at', 'deleted_by'])


# 4) Campaign
class Campaign(SoftDeleteMixin, models.Model):
    STATUS_CHOICES = (
        ('upcoming', 'Upcoming'),
        ('active', 'Active'),
        ('closed', 'Closed'),
    )
    VISIBILITY_CHOICES = (
        ('open', 'Open to all users'),
        ('invite', 'Invite Only'),
    )

    title = models.CharField(max_length=255)
    description = models.TextField()
    category = models.ForeignKey(Category, on_delete=models.SET_NULL, null=True)
    start_date = models.DateField()
    end_date = models.DateField()
    status = models.CharField(max_length=16, choices=STATUS_CHOICES, default='upcoming')
    visibility = models.CharField(max_length=10, choices=VISIBILITY_CHOICES, default='open')

    def __str__(self):
        return self.title


# 5) Idea
class Idea(SoftDeleteMixin, models.Model):
    STAGE_CHOICES = (
        ('suggest', 'Suggest'),
        ('define', 'Define'),
        ('refinement', 'Refinement'),
        ('evaluation', 'Evaluation'),
        ('pilot', 'Pilot'),
        ('portfolio', 'Portfolio'),
    )
    IDEA_TYPE_CHOICES = (
        ('PROCESS', 'Process'),
        ('PRODUCT', 'Product'),
        ('INNOVATION', 'Innovation'),
        ('SERVICES', 'Services'),
    )
    # Base info
    title = models.CharField(max_length=255)
    short_description = models.TextField(blank=True, null=True)  # ✅ این خط باید وجود داشته باشد
    description = models.TextField()
    author = models.ForeignKey(User, on_delete=models.CASCADE, related_name='submitted_ideas')
    campaign = models.ForeignKey(Campaign, on_delete=models.CASCADE, related_name='ideas')
    idea_type = models.CharField(max_length=50, choices=IDEA_TYPE_CHOICES)# ✅
    is_archived = models.BooleanField(default=False)
    archived_at = models.DateTimeField(null=True, blank=True)
    archived_by = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
        related_name="archived_ideas"
    )
    # OLD assessment fields (kept for UI compatibility)
    # utility = models.TextField(blank=True, null=True)
    # inventive_step = models.TextField(blank=True, null=True)
    # novelty_statement = models.TextField(blank=True, null=True)
    # trl = models.CharField(max_length=50, blank=True, null=True)

    # Suggest stage
    tags = models.ManyToManyField(Tag, blank=True)

    # NEW Concept Development fields (Phase 1)
    detailed_description = models.TextField(blank=True, null=True)
    strategic_alignment = models.TextField(blank=True, null=True)
    risks = models.TextField(blank=True, null=True)
    initial_people = models.TextField(blank=True, null=True)
    initial_abilities = models.TextField(blank=True, null=True)
    initial_departments = models.TextField(blank=True, null=True)
    feasibility = models.TextField(blank=True, null=True)
    trl = models.IntegerField(blank=True, null=True)

    # Interactions
    stage = models.CharField(max_length=20, choices=STAGE_CHOICES, default='suggest')
    likes = models.ManyToManyField(User, related_name='liked_ideas', blank=True)
    dislikes = models.ManyToManyField(User, related_name='disliked_ideas', blank=True)
    score = models.IntegerField(default=0)

    # Tracking
    created_at = models.DateTimeField(default=timezone.now, editable=False)
    last_stage_change = models.DateTimeField(default=timezone.now, editable=False)

    # Edit info
    edited = models.BooleanField(default=False)
    edited_at = models.DateTimeField(null=True, blank=True)
    edited_by = models.ForeignKey(
        User, null=True, blank=True,
        on_delete=models.SET_NULL,
        related_name='edited_ideas'
    )
    def save(self, *args, **kwargs):
        if self.idea_type:
            self.idea_type = self.idea_type.upper()  # 👈 همیشه با حروف بزرگ ذخیره شود
        super().save(*args, **kwargs)
    def __str__(self):
        return self.title
    final_score = models.FloatField(
        null=True, 
        blank=True, 
        help_text="نمره نهایی ارزیابی"
    )
    h_level = models.CharField(
        max_length=10,
        choices=[('H1', 'H1'), ('H2', 'H2'), ('H3', 'H3'), ('H4', 'H4')],
        null=True,
        blank=True,
        help_text="سطح H (ارزش‌گذاری)"
    )

# class EvaluationLog(models.Model):
#     idea = models.ForeignKey(Idea, on_delete=models.CASCADE, related_name='evaluation_logs')
#     evaluator = models.ForeignKey(User, on_delete=models.CASCADE)
#     criteria_scores = models.JSONField(default=dict)
#     final_score = models.DecimalField(max_digits=5, decimal_places=2)
#     h_level = models.CharField(max_length=10)
#     notes = models.TextField(blank=True)
#     evaluated_at = models.DateTimeField(auto_now_add=True)
    
#     class Meta:
#         ordering = ['-evaluated_at']

# 5.5) IdeaReview (NEW)
class IdeaReview(models.Model):
    idea = models.ForeignKey(Idea, on_delete=models.CASCADE, related_name='reviews')
    reviewer = models.ForeignKey(User, on_delete=models.CASCADE)

    stage = models.CharField(
        max_length=50,
        choices=[
            ('suggest', 'Suggest'),
            # ('concept_development', 'Concept Development'),
        ]
    )

    field_name = models.CharField(max_length=100)
    review_text = models.TextField()

    created_at = models.DateTimeField(default=timezone.now, editable=False)

    class Meta:
        ordering = ['-created_at']

    def __str__(self):
        return f"Review by {self.reviewer.username} on {self.idea.title} ({self.field_name})"
#5.5 activity log
class ActivityLog(models.Model):
    idea = models.ForeignKey(Idea, on_delete=models.CASCADE, related_name="activity_logs")
    previous_stage = models.CharField(max_length=50, null=True, blank=True)
    new_stage = models.CharField(max_length=50)
    notes = models.TextField(null=True, blank=True)
    changed_by = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True)
    changed_at = models.DateTimeField(default=timezone.now, editable=False)

    class Meta:
        ordering = ["-changed_at"]

# 6) Comment
class Comment(SoftDeleteMixin, models.Model):
    text = models.TextField()
    author = models.ForeignKey(User, on_delete=models.CASCADE)
    idea = models.ForeignKey(Idea, on_delete=models.CASCADE, related_name='comments')
    created_at = models.DateTimeField(default=timezone.now, editable=False)

    edited = models.BooleanField(default=False)
    edited_at = models.DateTimeField(null=True, blank=True)
    edited_by = models.ForeignKey(
        User, null=True, blank=True,
        on_delete=models.SET_NULL,
        related_name='edited_comments'
    )

    def __str__(self):
        return f"Comment by {self.author.username} on {self.idea.title}"


# 7) Attachment
class Attachment(models.Model):
    idea = models.ForeignKey(Idea, on_delete=models.CASCADE, related_name='attachments')
    file = models.FileField(upload_to='attachments/')
    file_type = models.CharField(max_length=50, blank=True)
    uploaded_at = models.DateTimeField(default=timezone.now, editable=False)

    def __str__(self):
        return f"Attachment for {self.idea.title}"


# 8) Notification
class Notification(models.Model):
    recipient = models.ForeignKey(User, on_delete=models.CASCADE, related_name='notifications')
    actor = models.ForeignKey(User, on_delete=models.CASCADE, related_name='+')
    verb = models.CharField(max_length=255)

    target_content_type = models.ForeignKey(
        ContentType,
        on_delete=models.CASCADE,
        null=True,
        blank=True
    )
    target_object_id = models.PositiveIntegerField(
        null=True,
        blank=True
    )
    target = GenericForeignKey(
        'target_content_type',
        'target_object_id'
    )

    unread = models.BooleanField(default=True)
    timestamp = models.DateTimeField(default=timezone.now, editable=False)

    class Meta:
        ordering = ['-timestamp']

    def __str__(self):
        if self.target:
            return f'{self.actor.username} {self.verb} {self.target}'
        return f'{self.actor.username} {self.verb}'


class RefinementQuestion(models.Model):
    key = models.CharField(max_length=100, unique=True)
    title = models.CharField(max_length=255)
    is_active = models.BooleanField(default=True)
    order = models.IntegerField(default=0)

    class Meta:
        ordering = ['order']

    def __str__(self):
        return self.title
class RefinementReview(models.Model):
    idea = models.ForeignKey(Idea, on_delete=models.CASCADE, related_name='refinement_reviews')
    reviewer = models.ForeignKey(User, on_delete=models.CASCADE)
    is_passed = models.BooleanField()
    created_at = models.DateTimeField(default=timezone.now, editable=False)

    def __str__(self):
        return f"Refinement - {self.idea_id} - {'PASSED' if self.is_passed else 'FAILED'}"
class RefinementAnswer(models.Model):
    review = models.ForeignKey(RefinementReview, on_delete=models.CASCADE, related_name='answers')
    question = models.ForeignKey(RefinementQuestion, on_delete=models.CASCADE)
    answer = models.BooleanField()
    comment = models.TextField(blank=True)
#Evaluation
class EvaluationCriterion(models.Model):
    key = models.CharField(
        max_length=50,
        unique=True,
        help_text="Unique identifier e.g. business_value"
    )
    title = models.CharField(
        max_length=200,
        help_text="Human readable title"
    )
    weight = models.FloatField(
        help_text="Final weight value (e.g. 0.25)"
    )
    is_active = models.BooleanField(
        default=True
    )

    created_at = models.DateTimeField(default=timezone.now, editable=False)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ["key"]

    def __str__(self):
        return f"{self.title} ({self.weight})"
class EvaluationResult(models.Model):
    idea = models.ForeignKey(Idea, on_delete=models.CASCADE, related_name="evaluations")
    evaluator = models.ForeignKey(User, on_delete=models.CASCADE)
    scores = models.JSONField()  # {criterion_id: score}
    final_score = models.FloatField()
    created_at = models.DateTimeField(default=timezone.now, editable=False)

    class Meta:
        unique_together = ('idea', 'evaluator')